/**
 *
 * \file        gateway.c
 *
 * \brief       handles device side of TCP/IP connection to the control system
 *
 * \author      Pete McCormick (copied from Jeff Paige)
 *
 * \date        5/28/2008
 *
 * \note        copied most recently from Cajita project
 *              The intention is to have this same module compile for DM and
 *              IDOCV...
 *
 * "gateway" and "cip client" are really the same thing in this context.
 *
 */

//#include "cresnet_env.h"
#include <stdio.h>
#include <string.h>
#include "nutypedefs.h"
#include "cip.h"
#include "ethernet.h"
#include "os.h"
#include "autodiscovery.h"
#include "string_utils.h"
//#include "autodiscovery.h"
//#include "safestring.h"
//#include "gateway_env.h"                // environment-specific includes
#include "gateway.h"
#include "cresnet_slave.h"
#include "Cresnet.h"
#include "console.h"
#include "errors.h"
#include "memorymanager.h"
#include "dm_field_debug.h"
#include "gateway_errors.h"

#ifdef EXTENDED_ETHERNET_SUPPORT
#include "DnsAutoDiscovery.h"
#endif

#if DEBUG
#define DmErrorPrintf if( IsDmFieldDebugIndexActive(DM_FIELD_ETHERNET_DEBUG_IDX) ) DmConsolePrintf
#else
#define DmErrorPrintf(...)
#endif

//this need to be the same priority as NetTcpReadTask to allow
//for resource initialization before NetTcpReadTask is scheduled
//a higher priority task will pre-empt a low priority task that created it
#define basicwebGATEWAY_PRIORITY (2)//aws- need to make this the same preprocessor as basicwebWEBSERVER_PRIORITY ( tskIDLE_PRIORITY + 2 )
////////////////////////////////////////////////////////////////////////////////

//#define TASK_STACK_BYTES                (1024*8)
//#define TASK_PRIORITY                   40
//#define CIP_DEBUG                       0x02

//#define CHECK_MAX_GW_SOCKET(socket)    if (socket > GatewayMaxSocket ) GatewayMaxSocket = socket;

#define GW_TCP_PORT                     41794

#define GW_TCP_NOTCONNECTED             0
#define GW_TCP_CONNECTED                1

#define HB_RESEND_TIMEOUT               10
#define HB_WAITING_TIMEOUT              20
#define HB_CONNECTING_TIMEOUT           10
#define HB_TRIES                        3

#define HB_WAITING_FOR_CONNECT          0
#define HB_WAITING_FOR_RESEND           1
#define HB_WAITING_FOR_RESPONSE         2
#define HB_WAITING_FOR_CONNECTION       3
#define HB_TERMINATED                   4

#define GW_CIP_STARTUP_MSEC             60000
#define GW_BUFFER_SIZE                  1024 //0x1000

// defines for Gateway mode byte
#define GATEWAY_ADD_MASTER_REQUEST      0x01
#define GATEWAY_WRITER_MODE_REQUEST     0x02
#define GATEWAY_CONNECTION_TIMEOUT      0x04
#define GATEWAY_ACTIVITY_RESETS_TIMEOUT 0x08
#define GATEWAY_PEER_TO_PEER            0x10
#define GATEWAY_CHECK_FOR_ACTIVATION    0x20
#define GATEWAY_CHECK_FOR_HEARTBEAT     0x40
#define GATEWAY_EXCLUSIVE               0x80  // terminate all other connections on IP/CID.

// defines for Gateway response flags
#define GATEWAY_RESPONSE_FLAGS_HEARTBEAT        0x01  // supports heartbeat and repeated digitals
//#define GATEWAY_RESPONSE_FLAGS_SEND_PROG_RDY    0x02  // supports the program ready message.

// defines for mode response in the gateway connect response.
#define ECONTROL2_TEMP_ACTIVATION           0x80  // a temporary activation of the Econtrol2 license.
#define AUTHENTICATION_REQUIRED             0x40  // authentication has been enabled for gateway connections.
#define GATEWAY_CONNECTION_SUCESS           0
#define GATEWAY_CONNECTION_REJECTED         2
#define GATEWAY_OUT_OF_LICENSES             4
#define ECONTROL2_ACTIVATION_FOUND          9  // the processor has a valid eControl2 activation key.
#define ECONTROL2_ACTIVATION_INVALID        10  // the processor has not been activated for eControl2.

// #defines for the Program Ready message
#define PROGRAM_READY_LOADING_FIRMWARE      0
#define PROGRAM_READY_NO_PROGRAM            1
#define PROGRAM_READY_PROGRAM_READY         2

////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    eDUMMY,                             // 0
    eCONNECT,                           // 1
    eCONNECT_RESPONSE,                  // 2
    eDISCONNECT,                        // 3
    eDISCONNECT_RESPONSE,               // 4
    eDATA,                              // 5
    eSERVER_PERSISTENCE,                // 6
    eDATA_STATUS,                       // 7
    eBUFFER_THRESHOLD,                  // 8
    eBUFFER_SIZE_REQUEST,               // 9
    eALTERNATE_CONNECT,                 // 10
    eAUTHENTICATE,                      // 11
    eAUTHENTICATE_RESPONSE,             // 12
    eHEARTBEAT,                         // 13
    eHEARTBEAT_RESPONSE,                // 14
    ePROGRAM_READY                      // 15
} EGatewayMessageTypes;

typedef enum
{
    eDISCONNECT_SUCCESS,
    eSERVER_CLOSED,
    eCIP_NOT_RESPONDING,
    eINVALID_HANDLER,
    eTIMED_OUT,
    eTIMING_ERROR
} EGatewayDisconnectErrors;

typedef enum
{
    eNORMAL_DISCONNECT,
    ePROGRAM_RESTART,
    eSYSTEM_REBOOT,
    eACTIVATION_EXPIRED,
    eDHCP_NEW_LEASE
} EGatewayDisconnectReasons;

typedef enum
{
    eBUFFER_NORMAL,
    ePAUSE_SENDING_DATA,
    eRESUME_SENDING_DATA,
    eBUFFER_FULL
} EGatewayBufferThresholdCodes;

////////////////////////////////////////////////////////////////////////////////

//typedef struct
//{
//    UINT8 stack[TASK_STACK_BYTES];
//    NU_TASK task;
//}GATEWAY;

//typedef struct
//{
//    UINT8 stack[TASK_STACK_BYTES];
//    NU_TASK task;
//}GATEWAYHEARTBEAT;

//#pragma pack(1)

typedef struct
{
    //Gateway Params
    UINT32         gwSocket;
    unsigned char  gwIPaddress[4];
    unsigned long  gwPort;
    unsigned long  gwStatus;
    unsigned short gwCID;
    unsigned short gwDID;
    unsigned short gwHandler;

    //Device Params
    //unsigned char  myIPaddress[4];
    //unsigned short myCID;
    unsigned long  myMID;

  // For Later USE ... NOT FULLY IMPLEMENTED
  // currently only supports 1 Stream / device
  unsigned char  bStream;  // Maps CIP Entry to Stream number

    //Heartbeat Params
    unsigned long  hbControl;
    long           hbTimeout;
    unsigned long  hbPayload;
    unsigned char  hbTries;
#ifdef GATEWAY_STATS
    unsigned long  hbSentCnt;
    unsigned long  hbSendErrCnt;
    unsigned long  hbRcvCnt;
    unsigned long  hbRcvErrCnt;

    unsigned long  connAttempts;
    unsigned long  connErrCnt;
    unsigned long  connCnt;
    unsigned long  gwServIpErrCnt;
#endif
    //Connection State Machine Params
    //struct
    //{
    //    unsigned long state;
    //}gwStateMachine;

    //Link List Params
    unsigned long  *pPrev;
    unsigned long  *pNext;

    //Reconnect Task
    //NU_TASK ReconnectTask;
    UINT32 TransmitSemaphore;
    void * timer;
    // An ethernet packet has a max payload of 1340 byte and can contain multiple
    // commands.  Therefore allow at least that much
    unsigned char GW_Buffer[GW_BUFFER_SIZE+360];
    int GW_Buffer_Size;
    int gotProgReady;
    BOOL cipStartupTimeout;
    UINT32 closeReason;
} Gateway_Server_Params;

//#pragma pack()

// structures for gateway messages.
// They must be packed because the structures mirror the packets as they are
// sent.
#pragma pack(1)
typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned char   Spare;
    unsigned char   PacketWithID;
} *pGATEWAY_MSG_HDR;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned long   DestIPAddress;
    unsigned short  DestCID;
    unsigned char   Mode;
    unsigned short  TimeOut;
    unsigned short  Type;
} *pGATEWAY_CONNECT_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Mode;
    unsigned char   Flags;
} GATEWAY_CONNECT_RESPONSE_MSG, *pGATEWAY_CONNECT_RESPONSE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Reason;
} *pGATEWAY_DISCONNECT_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned short  ErrorCode;
} *pGATEWAY_DISCONNECT_RESPONSE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Data[DM_MAX_CIP_MSG_LEN];
} GATEWAY_DATA_MSG, *pGATEWAY_DATA_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned char   RemainActive;
} *pGATEWAY_SERVER_PERSISTENCE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned short  Code;
} GATEWAY_BUFFER_THRESHOLD_MSG, *pGATEWAY_BUFFER_THRESHOLD_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Percent;
} *pGATEWAY_BUFFER_SIZE_REQUEST_MSG;
typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  DestCID;
    unsigned short  PortNumber;
    unsigned char   Mode;
    unsigned char   Flags;
    unsigned short  TimeOut;
    unsigned short  Type;
    char            HostName[63+1];
} *pGATEWAY_ALTERNATE_CONNECT_MSG;

// Added by AKN on 7/22/2003 5:57PM
typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Data[130];
} GATEWAY_AUTHENTICATE_MSG, *pGATEWAY_AUTHENTICATE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
    unsigned char   Mode;
} GATEWAY_AUTH_RESPONSE_MSG, *pGATEWAY_AUTH_RESPONSE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned short  Handler;
} GATEWAY_HEARTBEAT_MSG, *pGATEWAY_HEARTBEAT_MSG, GATEWAY_HEARTBEAT_RESPONSE_MSG, *pGATEWAY_HEARTBEAT_RESPONSE_MSG;

typedef struct
{
    unsigned char   MsgType;
    unsigned short  MsgLength;
    unsigned char   Status;
} PROGRAM_READY_MSG, *pPROGRAM_READY_MSG;

#pragma pack()

////////////////////////////////////////////////////////////////////////////////

int  ProcessGatewayPacket(Gateway_Server_Params* pGatewayParams, char *pcPktBuffer, int iPktSize);
int  SendGatewayResponse(Gateway_Server_Params* pGatewayParams, EGatewayMessageTypes eResponseType ,int iFlags ,int iMode ,int iHandler);
int  CloseGatewayConnection(Gateway_Server_Params* pGatewayParams, int reason, int code);
void CloseGatewayConnection_final(Gateway_Server_Params* pGatewayParams);
void ProcessGatewayData(Gateway_Server_Params* pGatewayParams, char *buffer);
//int  SendDataToGateway(MASTER_LIST_ENTRY* MasterNode, Gateway_Server_Params* pGatewayParams);
void GatewayUpdateEntry(MASTER_LIST_ENTRY* pEntry, BOOL hostnameChanged);
int  SendGatewayConnectUpdateRequest(MASTER_LIST_ENTRY* NewConnection, Gateway_Server_Params* pGatewayParams);
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
extern void  DMMidpointControllerRegisterCipMaster(MASTER_LIST_ENTRY* NewConnection);
#endif
////////////////////////////////////////////////////////////////////////////////

//GATEWAY Gateway;
//GATEWAYHEARTBEAT GatewayHeartbeat;

////////////////////////////////////////////////////////////////////////////////

void *    AllClearTimer;
#ifdef NEEDED
int g_bAllClearFilter;
extern int g_iAllClearCntr;
int g_iUpdateRequestCntr;
#define ALL_CLEAR_TIMER_FILTER_TIMEOUT      1000  // 10 seconds
int GatewayMaxSocket       = -1;
#endif // needed

////////////////////////////////////////////////////////////////////////////////
UINT16 ConvertTo16bitIpId(UINT16 IpId8Bit);

#define NO_16BIT_IPID 0xffff

UINT16 IpId16bit = NO_16BIT_IPID;
int GatewayConnectionCount =  0;
Gateway_Server_Params *gwServers = 0;

extern BOOL CresnetSupportSelectiveAllClear(void);

MASTER_LIST_ENTRY* GetMasterListEntry(MASTER_LIST_ENTRY * pMasterList, unsigned long handle);

extern BOOL CresnetDebug;
extern char CresnetDebugMessage[];

////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Jeff Paige
 *
 * \date        01/28/2005
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Gateway client heartbeat task
 *
 * \param       argc - count of string arguments passed in
 * \param       argv - array of string arguments
 *
 * \task        The Gateway Heartbeat Task sends and receives periodic CIP heartbeat messages
 *              to and from the control system/s.  There is only 1 of these tasks in the system.
 *              It is created at startup and runs forever.
 *
 */
void Gateway_Heartbeat_Task(UINT32 param)
{
    Gateway_Server_Params *gwThisServer = 0;
    int i;
    UINT8 buffer[10];
    pGATEWAY_HEARTBEAT_MSG pHeartbeat;
    INT32 status;
    UINT16 MsgLength;

    //while(1)
    //{
    if(NetGetIPAddress() == 0)
    {
        return;
    }

    gwThisServer = gwServers;
#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_HEARTBEAT))
    {
       DmConsolePrintf("Heartbeat task called, hbControl=%lu\r", gwThisServer->hbControl);
    }
#endif
    for(i=0;i<GatewayConnectionCount;i++)
    {
        if(gwThisServer->cipStartupTimeout)
        {
            gwThisServer->cipStartupTimeout = 0;
            gwThisServer->hbControl = HB_TERMINATED; //HB_WAITING_FOR_RESEND;
            CloseGatewayConnection(gwThisServer,ERROR_CIP_STARTUP_TIMEOUT,0);
        }
        else if(gwThisServer->hbControl == HB_WAITING_FOR_RESEND)
        {
            gwThisServer->hbTimeout--;

            if(gwThisServer->hbTimeout <= 0)
            {
                //Now wait for the response
                gwThisServer->hbControl = HB_WAITING_FOR_RESPONSE;
                gwThisServer->hbTimeout = HB_WAITING_TIMEOUT;

                if(gwThisServer->hbTries == 0)
                {
                    gwThisServer->hbPayload = HwGetMsec();
                    gwThisServer->hbTries   = HB_TRIES;
                    gwThisServer->hbTries--;
                }
                else
                {
                    gwThisServer->hbTries--;
                }

                //Build & send HB Packet Here!!!!
                pHeartbeat = (pGATEWAY_HEARTBEAT_MSG)buffer;
                pHeartbeat->MsgType              = eHEARTBEAT;
                pHeartbeat->MsgLength            = 2;
                MsgLength = pHeartbeat->MsgLength;
                pHeartbeat->Handler              = gwThisServer->gwHandler;
                //*(unsigned long*)&pHeartbeat[5]  = pGatewayParams->hbPayload;

                pHeartbeat->MsgLength = NATIVE2BE16(pHeartbeat->MsgLength);
                pHeartbeat->Handler = NATIVE2BE16(pHeartbeat->Handler);

                if(gwThisServer->gwStatus == GW_TCP_CONNECTED)
                {
                    OsLock(gwThisServer->TransmitSemaphore);
                    //status = NU_Send(gwThisServer->gwSocket, buffer, pHeartbeat->MsgLength + 3, 0);
                    //DmConsolePrintf("Sending heartbeat, socket=%lu\r", gwThisServer->gwSocket);
                    status = NetSendTcpData(gwThisServer->gwSocket, buffer, MsgLength + 3);
#ifdef CRESNET_GATEWAY_DEBUG
                    if(CresnetIsDebug(CRESNET_DEBUG_HEARTBEAT))
                    {
                        DmConsolePrintf("Heartbeat: NetSendTcpData returned %ld\r", status);
                    }
#endif
                    //NU_Release_Semaphore(&gwThisServer->TransmitSemaphore);
                    OsUnlock(gwThisServer->TransmitSemaphore);
                    if(status < 0)
                    {
#ifdef GATEWAY_STATS
                        gwThisServer->hbSendErrCnt++;
#endif
                        CloseGatewayConnection(gwThisServer,ERROR_SEND_FAIL,status);
                    }
                    else if(status < MsgLength + 3)
                    {
#ifdef GATEWAY_STATS
                        gwThisServer->hbSendErrCnt++;
#endif
                        DmErrorPrintf("GWHT: only sent %ld of %u bytes\r\n", status, MsgLength + 3);
                    }
#ifdef GATEWAY_STATS
                    else
                    {
                        gwThisServer->hbSentCnt++;
                    }
#endif
                }
#ifdef GATEWAY_STATS
                else
                {
                    gwThisServer->hbSendErrCnt++;
                }
#endif
            }
        }
        else if(gwThisServer->hbControl == HB_WAITING_FOR_RESPONSE)
        {
            gwThisServer->hbTimeout--;

            if(gwThisServer->hbTimeout <= 0)
            {
                if(gwThisServer->hbTries > 0)
                {
                    gwThisServer->hbControl = HB_WAITING_FOR_RESEND;
                }
                else
                {
                    //No heartbeat response in over 10 seconds
                    DmErrorPrintf("GW Heartbeat timeout\r\n");

                    CloseGatewayConnection(gwThisServer, ERROR_HB_TIMEOUT, 0);

                    gwThisServer->hbControl = HB_TERMINATED; //HB_WAITING_FOR_RESEND;
                    gwThisServer->hbTimeout = HB_RESEND_TIMEOUT;
                    gwThisServer->hbPayload = 0;
                    gwThisServer->hbTries   = 0;
                }
            }
        }
#ifndef GWTIMER
      else if(gwThisServer->hbControl == HB_WAITING_FOR_CONNECTION)
      {
        gwThisServer->hbTimeout--;

        if(gwThisServer->hbTimeout <= 0)
        {
          if(gwThisServer->hbTries > 0)
          {
            gwThisServer->hbControl = HB_WAITING_FOR_CONNECTION;
            gwThisServer->hbTimeout = HB_CONNECTING_TIMEOUT;
          }
          else
          {
            DmErrorPrintf("GW Connection timeout\r\n");

            CloseGatewayConnection(gwThisServer, ERROR_HB_TIMEOUT, 0);

            gwThisServer->hbControl = HB_TERMINATED; //HB_WAITING_FOR_RESEND;
            gwThisServer->hbTimeout = HB_RESEND_TIMEOUT;
            gwThisServer->hbPayload = 0;
            gwThisServer->hbTries   = 0;
          }
        }
      }
#endif

        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
        //}
        //NU_Sleep(100); //Sleep for 1 second
//        HwDelayMsec(1000);
    //}
}
#ifdef GWTIMER
static void GatewayCipStartupTimerExpiration(UINT32 param)
{
    Gateway_Server_Params * gwThisServer = (Gateway_Server_Params *)param;
    gwThisServer->cipStartupTimeout = 1;
}

void Gateway_Stop_Cip_Startup_Timer(Gateway_Server_Params * gwThisServer)
{
    //if(NU_Control_Timer(&gwThisServer->timer, NU_DISABLE_TIMER) != NU_SUCCESS)
    //{
    //    DmErrorPrintf("GATEWAY: could not disable timer");
    //}
    OsCancelTimer(gwThisServer->timer);
    gwThisServer->cipStartupTimeout = 0;
}

void Gateway_Start_Cip_Startup_Timer(Gateway_Server_Params * gwThisServer)
{
    //UINT32 ticks;

    //if(NU_Control_Timer(&gwThisServer->timer, NU_DISABLE_TIMER) != NU_SUCCESS)
    //{
    //    DmErrorPrintf("GATEWAY: could not disable timer");
    //}
    OsCancelTimer(gwThisServer->timer);

    gwThisServer->cipStartupTimeout = 0;

    //ticks = HwMsToTicks(GW_CIP_STARTUP_MSEC);
    //
    //if(NU_Reset_Timer(&gwThisServer->timer, GatewayCipStartupTimerExpiration,
    //        ticks, 0,
    //        NU_ENABLE_TIMER) != NU_SUCCESS)
    //{
    //    DmErrorPrintf("GATEWAY: Could not restart timer!");
    //}
    OsUpdateTimer(gwThisServer->timer, GW_CIP_STARTUP_MSEC);
}
#endif
void Gateway_Got_Connected(Gateway_Server_Params *gwThisServer)
{
    gwThisServer->gwStatus = GW_TCP_CONNECTED;
    gwThisServer->closeReason = ERROR_NOT_CLOSING;
#ifdef GATEWAY_STATS
    gwThisServer->connCnt++;
#endif
#ifndef GWTIMER
        //if we don't log in after 10 seconds, retry...
    gwThisServer->hbControl = HB_WAITING_FOR_CONNECTION; //HB_WAITING_FOR_RESEND;
    gwThisServer->hbTimeout = HB_CONNECTING_TIMEOUT;
    gwThisServer->hbPayload = 0;
    gwThisServer->hbTries   = 0;
#else
    Gateway_Start_Cip_Startup_Timer(gwThisServer);
#endif
}

INT32 Gateway_Receive_Data(UINT32 param, UINT8 * pData, UINT32 dataBytes)
{
    Gateway_Server_Params *gwThisServer = (Gateway_Server_Params *)param;
    //INT32 result;
#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_PKTS))
    {
        DmConsolePrintf("Received %lu bytes from control system\r", dataBytes);
        DmConsoleHexDumpPacket((UINT8*)pData, (UINT16)dataBytes, "From TCP:", 1);
    }
#endif
    if(dataBytes)
    {
      ProcessGatewayPacket(gwThisServer, (char*)pData, dataBytes);
    }
    else// zero data received now cleanly closes gateway connection
    {
       CloseGatewayConnection(gwThisServer,ERROR_RECV_FAIL,dataBytes);
    }
    //return closing flag so the read task can suspend itself
    if( gwThisServer->closeReason )
       return -1;

    return 0;
}

INT32 GatewayNewConnection(UINT32 param, UINT32 remoteIp, UINT16 remotePort, UINT32 socket)
{
    Gateway_Server_Params *gwThisServer = (Gateway_Server_Params *)param;
    UINT32 inst = 0;
    MASTER_LIST_ENTRY * pEntry = MasterListLockInst(inst);
    //MASTER_LIST_ENTRY * pMasterList = GetMasterListEntry(pEntry, gwThisServer->myMID);

    // need to save the socket so we can send data back to the remote endpoint
    gwThisServer->gwSocket = socket;
    if(pEntry)
    {
        pEntry->socket = socket;
    }
    MasterListUnlockInst(inst);

    return 0;
}

/**
 * \author      Jeff Paige
 *
 * \date        01/28/2005
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Gateway client reconnect task
 *
 * \param       argc - count of string arguments passed in
 * \param       argv - array of string arguments
 *
 * \task        The Gateway Reconnect Task maintains a TCP connection to it gateway server.
 *              There is one of this task for every gateway that this client has a
 *              relationship with.  The task is created when a gateway entry is added, and runs forever.
 *
 */
void Gateway_Reconnect_Task(UINT32 param)
{
    //struct addr_struct servaddr;
    Gateway_Server_Params *gwThisServer = 0;
    int result;
    UINT32 IP_Address;
    MASTER_LIST_ENTRY* pEntry;
    //int OptVal = 1;
#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_RECONNECT))
    {
        DmConsolePrintf("Gateway_Reconnect_Task called\r");
    }
#endif
    gwThisServer = (Gateway_Server_Params*)param;

    if(gwThisServer == NULL)
    {
        DmErrorPrintf("GW: No server!");
        return;
    }

    //while(1)
    //{
        if((gwThisServer->gwStatus != GW_TCP_CONNECTED) && (NetGetIPAddress() != 0))
        {
            //if close pending
            if( gwThisServer->closeReason )
            {
               CloseGatewayConnection_final(gwThisServer);
               gwThisServer->closeReason = ERROR_NOT_CLOSING;
            }
            //if(gwThisServer->gwSocket >= 0)
            //{
                // gwIPaddress will always be zero for autodiscovery
                // otherwise it will be set and work as before
                IP_Address = *((UINT32*)gwThisServer->gwIPaddress);
                if(IP_Address == 0)
                {
                    // need to resolve it
                    pEntry = MasterListLockInst(0);
                    pEntry = GetMasterListEntry(pEntry, gwThisServer->myMID);
                    if(pEntry)
                    {
                        // use host name in master list entry
                        // to try to resolve via autodiscovery dns

#ifndef EXTENDED_ETHERNET_SUPPORT
                        AutodiscoveryGetHostByName(pEntry->SiteName, &IP_Address);
#else
                        // Resolve hostname using CDnsAutoDiscovery
                        if (g_pDnsAutoDiscovery != NULL)
                          g_pDnsAutoDiscovery->GetHostByName(pEntry->SiteName,
                                                             sizeof(pEntry->SiteName),
                                                             (UINT8*)&IP_Address,
                                                             sizeof(IP_Address));


#endif
                        // if this fails, IP_Address will still be zero
                    }
                    MasterListUnlockInst(0);
                }
                if(IP_Address == 0)
                {
#ifdef GATEWAY_STATS
                    gwThisServer->gwServIpErrCnt++;
#endif
                }
                else
                {
                    // we have a socket already
                    //servaddr.family = 2;
                    //memcpy(&servaddr.id, &IP_Address, 4);
                    //servaddr.port = gwThisServer->gwPort;
#ifdef GATEWAY_STATS
                    gwThisServer->connAttempts++;
#endif
                    gwThisServer->gotProgReady = 0;
                    //result = NU_Connect(gwThisServer->gwSocket, &servaddr, 0);
                    //result = NetListenOnTcpPort(gwThisServer->gwPort, 1,
                    //   NULL, Gateway_Receive_Data, (UINT32)gwThisServer,
                    //   &gwThisServer->gwSocket);
#ifdef CRESNET_GATEWAY_DEBUG
                    if(CresnetIsDebug(CRESNET_DEBUG_RECONNECT))
                    {
                        DmConsolePrintf("Trying to connect to %lx\r", NATIVE2BE32(IP_Address));
                    }
#endif
                    result = NetTcpConnect(IP_Address, gwThisServer->gwPort, GatewayNewConnection,
                        Gateway_Receive_Data, (UINT32)gwThisServer);
#ifdef CRESNET_GATEWAY_DEBUG
                    if(CresnetIsDebug(CRESNET_DEBUG_RECONNECT))
                    {
                        DmConsolePrintf("NetTcpConnect returned %d\r", result);
                    }
#endif

                    if(result < 0)
                    {
#ifdef GATEWAY_STATS
                        gwThisServer->connErrCnt++;
#endif
                        CloseGatewayConnection(gwThisServer,ERROR_CONNECT_FAIL,result);
                        CloseGatewayConnection_final(gwThisServer);
                        gwThisServer->closeReason = ERROR_NOT_CLOSING;
                    }
                    else
                    {
                        Gateway_Got_Connected(gwThisServer);
                    }
                }
                // otherwise don't bother to connect
            }
            else // already connected
            {
#ifdef CRESNET_GATEWAY_DEBUG
                if(CresnetIsDebug(CRESNET_DEBUG_RECONNECT))
                    DmConsolePrintf("Already connected to gateway, nothing to do...\r");
#endif
            }
            //if(gwThisServer->gwSocket < 0)
            //{
                //There is no Socket, Get one
            //    gwThisServer->gwSocket = NU_Socket(NU_FAMILY_IP,NU_TYPE_STREAM, 0);
                // if NU_Socket fails, we will automatically try again
                //if ( NU_Setsockopt(gwThisServer->gwSocket, IPPROTO_TCP,
                //    TCP_NODELAY, &OptVal, sizeof(int)) != NU_SUCCESS)
                //{
                //    DmErrorPrintf("GW: unable to disable Nagle algorithm.");
                //}
            //}
        //}
        //NU_Sleep(500);
//        HwDelayMsec(5000);
    //}
}

/********************************************************
 * Function : InitGatewayEntryData
 ********************************************************/
static int InitGatewayEntryData(void)
{
    //Make sure conn count is zero...
    GatewayConnectionCount = 0;

    //We need at least one struct to start with
    gwServers = (Gateway_Server_Params*)OsAllocMemory(sizeof(Gateway_Server_Params));
    if(gwServers)
    {
        memset(gwServers,0x00,sizeof(Gateway_Server_Params));
        return 0;
    }
    else
        return -1;
}


/**
 * \author      Jeff Paige
 *
 * \date        01/28/2005
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Gateway client task
 *
 * \param       argc - count of string arguments passed in
 * \param       argv - array of string arguments
 *
 * \task        The Gateway (Client) Task initiates the creation of the
 *              Gateway Heartbeat and Reconnect Tasks.  It also waits on
 *              TCP sockets and handles CIP packets received from the
 *              control system/s. There is one of this task in the entire system.
 *              This task is created at startup and runs forever.
 *
 */
UINT8 GatewayHeartTaskExtra[OS_TASK_EXTRA_BYTES];
void Gateway_Task(UINT32 param)
{
    //struct addr_struct servaddr;
    //FD_SET      readfs,writefs,exceptfs;
    int result; //, fdSetCount;
    //Gateway_Server_Params *gwThisServer = 0;
    MASTER_LIST_ENTRY* pEntry;
    unsigned short NumEntries = 0;
    //INT32 status;
    UINT8 * extra = GatewayHeartTaskExtra;

#ifdef CRESNET_GATEWAY_DEBUG
    CresnetSetDebug(CRESNET_DEBUG_CIP_PKTS,1);

    if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
    {
        DmConsolePrintf("Gateway_Task called\r");
    }
#endif
    //StartupWait();

    // want to go ahead and create tasks even if there are no entries right now

    result = InitGatewayEntryData();
#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
    {
        DmConsolePrintf("InitGatewayEntryData returned %d\r", result);
    }
#endif
    if(result != 0)
        return;

    // search through the entire list
    pEntry = MasterListLockInst(0);
    while( (pEntry != NULL) )
    {
        if (pEntry->IP_Address != CIP_DELETED_ENTRY)
          {
            // PEM - all entries now connect by tcp
            //if(pEntry->Flags & CIP_CONNECT_BY_TCP)
            //{
                result = AddGatewayEntry(pEntry, pEntry->handle);
#ifdef CRESNET_GATEWAY_DEBUG
                if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
                {
                    DmConsolePrintf("AddGatewayEntry returned %d\r", result);
                }
#endif
            //}
            NumEntries++;
          }
        pEntry = pEntry->Next;
    }
    MasterListUnlockInst(0);

    // no entries, so force to offline
    if(NumEntries == 0)
    {
        MyCipOfflineEvent();
    }

//#ifdef CREATE_OTHER_TASKS
    //Create Heartbeat Task
    //status = NU_Create_Task(&GatewayHeartbeat.task, "GTWHBT", Gateway_Heartbeat_Task, 0, NU_NULL,
    //        GatewayHeartbeat.stack, sizeof(GatewayHeartbeat.stack), TASK_PRIORITY, 0, NU_PREEMPT, NU_START);
#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
    {
        DmConsolePrintf("Creating heartbeat task...\r");
    }
#endif
    OsCreateNamedTask(Gateway_Heartbeat_Task, 1000, 0, &param, extra, "GwH");
//#endif

    //if (status != 0)
    //{
    //    DmSystemError(FATAL_APPINIT);
    //}

    //while (1)      // loop forever
    //{
    //    gwThisServer = gwServers;
    //    for(i=0;i<GatewayConnectionCount;i++)
    //    {
    //        CHECK_MAX_GW_SOCKET(gwThisServer->gwSocket);
    //        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    //    }
    //    NU_FD_Init(&readfs);
    //    //Set FD for each socket in the list
    //    fdSetCount = 0;
    //    gwThisServer = gwServers;
    //    for(i=0;i<GatewayConnectionCount;i++)
    //    {
    //        if(gwThisServer->gwStatus == GW_TCP_CONNECTED)
    //        {
    //            NU_FD_Set(gwThisServer->gwSocket,&readfs);
    //            fdSetCount++;
    //        }
    //        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    //    }

    //    if(fdSetCount == 0)
    //    {
    //        // try not doing a select if there are no sockets
    //        NU_Sleep(100);
    //        continue;
    //    }

    //    if (NU_Select(GatewayMaxSocket + 1,&readfs,&writefs,&exceptfs,ETHER_SELECT_TIMEOUT) == NU_SUCCESS)
    //    {
    //        // at this point, a connection was made on some socket
    //        gwThisServer = gwServers;
    //        for(i=0;i<GatewayConnectionCount;i++)
    //        {
    //            if(gwThisServer->gwStatus == GW_TCP_CONNECTED)
    //            {
    //                if (NU_FD_Check(gwThisServer->gwSocket,&readfs) == NU_TRUE)
    //                {
    //                   //Read the packet
    //                    result = NU_Recv(gwThisServer->gwSocket, (char*)GW_Buffer, GW_BUFFER_SIZE, 0);
    //                    if(result > 0)
    //                        ProcessGatewayPacket(gwThisServer,(char*)GW_Buffer,result);
    //                    else if(result == NU_NOT_CONNECTED)
    //                        CloseGatewayConnection(gwThisServer,ERROR_RECV_FAIL,result);
    //                }
    //            }
    //            gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    //        }
    //    }
        //If no connections are open, then NU_Select will return without suspending???, so sleep to keep from
        //blocking the whole system!!!!
        //if(fdSetCount == 0)
        //{
        //    NU_Sleep(100);
        //}
    //}
}

/********************************************************
 * Function : CloseGatewayConnection
 ********************************************************/
int CloseGatewayConnection(Gateway_Server_Params* pGatewayParams, int reason, int code)
{
    switch(reason)
    {
        case ERROR_SEND_FAIL:
            DmErrorPrintf("GW Closed Socket - Send Fail %d\r\n",code);
            break;
        case ERROR_REMOTE_CLOSE:
            DmErrorPrintf("GW Closed Socket - Remote Closed\r\n");
            break;
        case ERROR_RECV_FAIL:
            DmErrorPrintf("GW Closed Socket - Recv Fail %d\r\n",code);
            break;
        case ERROR_HB_TIMEOUT:
            DmErrorPrintf("GW Closed Socket - HB Timeout\r\n");
            break;
        case ERROR_CONNECT_FAIL:
            DmErrorPrintf("GW Closed Socket - Connection Failed %d\r\n",code);
            break;
        case ERROR_LOGIN_FAIL:
            DmErrorPrintf("GW Closed Socket - Login Failed\r\n");
            break;
        case ERROR_MASTER_LIST_DEL:
            DmErrorPrintf("GW Closed Socket - IP Table entry deleted\r\n");
            break;
        case ERROR_MASTER_LIST_CHANGE:
            DmErrorPrintf("GW Closed Socket - IP Table entry changed\r\n");
            break;
        case ERROR_PROGRAM_NO_PROGRAM:
            DmErrorPrintf("GW Closed Socket - No Program\r\n");
            break;
        case ERROR_TX_PIPE_TIMEOUT:
            DmErrorPrintf("GW Closed Socket - Transmit Pipe Timeout\r\n");
            break;
        case ERROR_CIP_STARTUP_TIMEOUT:
            DmErrorPrintf("GW Closed Socket - CIP Startup Timeout\r\n");
            break;
        default:
            DmErrorPrintf("GW Closed Socket - reason %d, code %d\r\n", reason, code);
            reason = ERROR_UNKNOWN_REASON;
            break;

    }

    pGatewayParams->closeReason |= reason;
    pGatewayParams->gwStatus = GW_TCP_NOTCONNECTED;

    return 0;

}
void CloseGatewayConnection_final(Gateway_Server_Params* pGatewayParams)
{
    MASTER_LIST_ENTRY*  pEntry;
    char temp[80];
    short int prevStatus;

    //close the socket and delete the read task
    if (pGatewayParams->closeReason != ERROR_CONNECT_FAIL)
    {
       NetClose(pGatewayParams->gwSocket,0);
    } //if (reason != ERROR_CONNECT_FAIL)already cleaned up

    pGatewayParams->gwSocket = -1;
    pGatewayParams->gwStatus = GW_TCP_NOTCONNECTED;

    // MNT - 1/9/2009 - Changes made by Pete
    // clear buffer
    pGatewayParams->GW_Buffer_Size = 0;

    if(pGatewayParams->closeReason != ERROR_MASTER_LIST_DEL)
    {
        pEntry = MasterListLockInst(0);
        pEntry = GetMasterListEntry(pEntry, pGatewayParams->myMID);
        if(pEntry)
        {
            prevStatus = pEntry->LinkStatus;
            if(prevStatus != CIP_OFFLINE)
            {
                pEntry->LinkStatus = CIP_OFFLINE;
                //  4/9/07 - added by HS
                CIPOfflineEvent(pEntry);
                if(pEntry->IP_Address)
                {
                    ConvertIPAddressToString(pEntry->IP_Address, temp);
                }
                else
                {
                    _strncpy(temp, pEntry->SiteName, sizeof(temp));
                }
                DmErrorPrintf("GATEWAY: offline with %s\r\n", temp);
            }
        }
        MasterListUnlockInst(0);
    }

    return ;
}

void CloseGatewayConnectionByHandle(UINT32 handle)
{
    Gateway_Server_Params *gwThisServer = 0;
    int i;

    gwThisServer = gwServers;
    for(i=0; i<GatewayConnectionCount; i++)
    {
        if(gwThisServer->myMID == handle)
        {
            CloseGatewayConnection(gwThisServer, ERROR_MASTER_LIST_DEL, 0);
        }
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
}

/********************************************************
 * Function : CloseAllGatewayConnections
 ********************************************************/
void CloseAllGatewayConnections(void)
{
    MASTER_LIST_ENTRY*  pEntry;
    Gateway_Server_Params *gwThisServer = 0;
    int i;

    gwThisServer = gwServers;
    for(i=0;i<GatewayConnectionCount;i++)
    {
        NetClose(gwThisServer->gwSocket,0);
        gwThisServer->gwSocket = -1;
        gwThisServer->gwStatus = GW_TCP_NOTCONNECTED;

        pEntry = MasterListLockInst(0);
        pEntry = GetMasterListEntry(pEntry, gwThisServer->myMID);
        if(pEntry)
        {
            pEntry->LinkStatus = CIP_OFFLINE;
            //  4/9/07 - added by HS
            CIPOfflineEvent(pEntry);
        }
        MasterListUnlockInst(0);

        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
}

void AbortAllGatewayConnections(void)
{
    MASTER_LIST_ENTRY*  pEntry;
    Gateway_Server_Params *gwThisServer = 0;
    int i;

    gwThisServer = gwServers;
    for(i=0;i<GatewayConnectionCount;i++)
    {
        if(NetClose(gwThisServer->gwSocket,1) != 0)
        {
            DmErrorPrintf("Error aborting gateway socket!\r\n");
        }
        gwThisServer->gwSocket = -1;
        gwThisServer->gwStatus = GW_TCP_NOTCONNECTED;

        pEntry = MasterListLockInst(0);
        pEntry = GetMasterListEntry(pEntry, gwThisServer->myMID);
        if(pEntry)
        {
            pEntry->LinkStatus = CIP_OFFLINE;
            //  4/9/07 - added by HS
            CIPOfflineEvent(pEntry);
        }
        MasterListUnlockInst(0);

        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
}

/********************************************************
 * Function :  ProcessGatewayPacket
 ********************************************************/
int ProcessGatewayPacket(Gateway_Server_Params* pGatewayParams, char *pcPktBuffer, int iPktBufferSize)
{
    MASTER_LIST_ENTRY*  MasterPtr;
    pGATEWAY_MSG_HDR pDataHeader;
    //unsigned short CIP_ID;
    //int result;
    unsigned char *pData = (unsigned char*)pGatewayParams->GW_Buffer;
    int iBytes = pGatewayParams->GW_Buffer_Size + iPktBufferSize;
    int iMsgSize = 0;
    int i;

    //Copy new data into process buffer
    memcpy( (char*)(pGatewayParams->GW_Buffer + pGatewayParams->GW_Buffer_Size),pcPktBuffer,iPktBufferSize);

    //Process the whole buffer, there may be more than one packet and perhaps a part of one
    while(iBytes > 0)
    {
        pDataHeader = (pGATEWAY_MSG_HDR)pData; //Point Header at start of next packet

        pDataHeader->MsgLength = BE2NATIVE16(pDataHeader->MsgLength);   // MsgLength needs to be endian swapped

        // MNT - 3/31/2009 - Bug fix taken from the touch panel code
        // Max size = serial + 3 bytes for header (258)
        if( (iBytes > 3) && ((pDataHeader->MsgType > ePROGRAM_READY) || (pDataHeader->MsgLength >= 259)) )

        {
            DmErrorPrintf("GW_Buffer Message Error %X %X\r\n",pDataHeader->MsgType, pDataHeader->MsgLength);
            DmErrorPrintf("[%X][%X][%X][%X][%X][%X][%X][%X][%X][%X]\r\n",
                    pData[0],pData[1],pData[2],pData[3],pData[4],pData[5],pData[6],pData[7],pData[8],pData[9]);

            while(iBytes) //Move to the next byte in the buffer
            {
                pData++;
                iBytes--;

                pDataHeader = (pGATEWAY_MSG_HDR)pData;
                // MNT - 1/9/2009 - Bug fix made by Pete
                pDataHeader->MsgLength = BE2NATIVE16(pDataHeader->MsgLength);
                // MNT - 3/31/2009 - Bug fix taken from the touch panel code
                // Max size = serial + 3 bytes for header (258)
                if( (pDataHeader->MsgType <= ePROGRAM_READY) && (pDataHeader->MsgLength < 259) )
                    break;
            }
        }

        iMsgSize = pDataHeader->MsgLength + 3;
        if( (iBytes > 3) && (iBytes >= iMsgSize) ) //Do we have at least 4 bytes and a whole packet?

        {
            //if(CresnetDebug & CIP_DEBUG)
            //{
            //    cp = CresnetDebugMessage;
            //    cp += sprintf(cp,"GW Rx:");
            //    i = 0;
            //    do
            //    {
            //        cp += sprintf(cp," %02X",*(pData+i));
            //        i++;
            //    } while (i < iMsgSize);
            //    sprintf(cp,"\r\n");
            //    //WriteDebugData(DebugMessage);
            //    DmConsolePrintf(CresnetDebugMessage);
            //}
#ifdef CRESNET_GATEWAY_DEBUG
            if(CresnetIsDebug(CRESNET_DEBUG_CIP_MSGS))
            {
                DmConsolePrintf("Received gateway packet type %d\r", pDataHeader->MsgType);
            }
#endif
            switch ( (EGatewayMessageTypes)pDataHeader->MsgType )
            {
                case eCONNECT_RESPONSE : //If ok, then update whatever so that packets get sent to the server
                    if(pData[5] == 0)
                    {
                        pGatewayParams->gwHandler = *(unsigned short*)&pData[3];
                        //CIP_ID = pGatewayParams->gwCID;
                        //IP_Address = *(unsigned long *)&pGatewayParams->gwIPaddress[0];
                        // IP_Address will be zero for autodiscovery, in that case
                        // SiteName will be set
                        MasterPtr = MasterListLockInst(0);
                        MasterPtr = GetMasterListEntry(MasterPtr, pGatewayParams->myMID);
                        if(MasterPtr != NULL)
                        {
                            if (MasterPtr->LinkStatus != CIP_ONLINE )
                            {
                                CIPOnlineEvent(MasterPtr);

                                MasterPtr->LinkStatus = CIP_ONLINE;
                                //Mark updateRequest as sent to we don't trigger a UDP initial connect
                                //MasterPtr->m_bUpdateRequestSent = 1;
                                SendGatewayConnectUpdateRequest(MasterPtr, pGatewayParams);
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
                                DMMidpointControllerRegisterCipMaster(MasterPtr);
                                
#endif
                                //result = SendGatewayConnectUpdateRequest(MasterPtr, pGatewayParams);
                                //if(result != 0)
                                //    CheckForQueuedMessages(MasterPtr);
                            }
                        }
                        MasterListUnlockInst(0);
#ifdef GWTIMER
                        // stop the startup timeout timer -
                        // heartbeats
                        Gateway_Stop_Cip_Startup_Timer(pGatewayParams);
#endif
                        if(pData[6] & GATEWAY_RESPONSE_FLAGS_HEARTBEAT)//Support for Heartbeat msgs //Start Heartbeat
                        {
                            pGatewayParams->hbControl = HB_WAITING_FOR_RESEND;
                            pGatewayParams->hbTimeout = HB_RESEND_TIMEOUT;
                            pGatewayParams->hbPayload = 0;
                            pGatewayParams->hbTries   = 0;
                        }
                        else
                        {
                            DmErrorPrintf("GATEWAY: Connect Response: No Heartbeats %x\r\n", pData[6]);
                        }
                    }
                    else //If not ok, then close the connection
                    {
                        CloseGatewayConnection(pGatewayParams,ERROR_LOGIN_FAIL,0);
                    }
                    break;
                case eDISCONNECT :
                    // fall through
                case eDISCONNECT_RESPONSE : //We got disconnected, or could no log into an older server.
                    // don't think we should ever get a eDISCONNECT_RESPONSE
                    CloseGatewayConnection(pGatewayParams,ERROR_REMOTE_CLOSE,0);
                    break;
                case ePROGRAM_READY :       //We got a program ready. Log in
                    // PROGRAM_READY_LOADING_FIRMWARE is zero so the following check
                    // will always fail:
                    //if(pData[3] & PROGRAM_READY_LOADING_FIRMWARE)//Program Ready, Log in
                    //    CloseGatewayConnection(pGatewayParams,ERROR_LOGIN_FAIL,0);
                    if(pData[3] & PROGRAM_READY_NO_PROGRAM)
                    {
                        CloseGatewayConnection(pGatewayParams,ERROR_PROGRAM_NO_PROGRAM,0);
                    }
                    if(pData[3] & PROGRAM_READY_PROGRAM_READY)
                    {
                        if(!pGatewayParams->gotProgReady)
                        {
                            pGatewayParams->gotProgReady = 1;
                            SendGatewayResponse(pGatewayParams, eALTERNATE_CONNECT, 0 , 0 , 0);
                        }
                        else
                        {
                            DmErrorPrintf("GATEWAY: ignoring extra Program Ready msg\r\n");
                        }
                    }
                    else
                    {
                        DmErrorPrintf("GATEWAY: got Program (Not) Ready %x\r\n", pData[3]);
                    }
                    break;
                case eDATA :                    //Send to GWAY handler
                    ProcessGatewayData(pGatewayParams,(char*)pData);
                    break;
                case eHEARTBEAT_RESPONSE :      //Reset the HB timeout timer since the connection is still alive
#ifdef CRESNET_GATEWAY_DEBUG
                    if(CresnetIsDebug(CRESNET_DEBUG_HEARTBEAT))
                    {
                        DmConsolePrintf("Heartbeat: Got response\r");
                    }
#endif
                    pGatewayParams->hbControl = HB_WAITING_FOR_RESEND;
                    pGatewayParams->hbTimeout = HB_RESEND_TIMEOUT;
                    pGatewayParams->hbPayload = 0;
                    pGatewayParams->hbTries   = 0;
#ifdef GATEWAY_STATS
                    pGatewayParams->hbRcvCnt++;
#endif
                    break;
                case eDUMMY:
                    break;
                default:
                    break;
            }
            //Reduce iBytes by the packet size and reduce iBytes
            pData += iMsgSize;
            iBytes -= iMsgSize;
        }
        else  //Store what's left and set the pointers and buffer sizes...
        {
            // restore the original endian byte order for MsgLength
            // since it will be byte-swapped on next invocation of this function
            pDataHeader->MsgLength = BE2NATIVE16(pDataHeader->MsgLength);

            for(i=0;i<iBytes;i++)
                pGatewayParams->GW_Buffer[i] = pData[i];

            pGatewayParams->GW_Buffer_Size = iBytes;
            break;  //Exit the while loop
        }
    }

    if(iBytes <= 0) //If we processed all the data in the buffer
        pGatewayParams->GW_Buffer_Size = 0;

    return 0;
}

/********************************************************
 * Function : SendGatewayResponse
 ********************************************************/
int SendGatewayResponse(Gateway_Server_Params* pGatewayParams, EGatewayMessageTypes eResponseType ,int iFlags ,int iMode ,int iHandler)
{
    UINT8 pGatewayResponse[DM_MAX_CIP_MSG_LEN];
    int iDataLength = 0;
    INT32 bytes_sent;
    pGATEWAY_ALTERNATE_CONNECT_MSG pResponse;
    MASTER_LIST_ENTRY *pEntry;

    memset(pGatewayResponse,0,DM_MAX_CIP_MSG_LEN);

#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_MSGS))
    {
        DmConsolePrintf("Sending packet type %u to control system...\r", eResponseType);
    }
#endif

    pResponse = (pGATEWAY_ALTERNATE_CONNECT_MSG)pGatewayResponse;

    switch ( eResponseType )
    {
        case eALTERNATE_CONNECT : //Use the alt connect to cover all the bases?
            pResponse->MsgType = eResponseType;
            pResponse->MsgLength = 10;
            //pResponse->DestIPAddress = *(unsigned long*)&pGatewayParams->myIPaddress[0];
            pResponse->DestCID = pGatewayParams->gwCID;
            pResponse->PortNumber = pGatewayParams->gwPort; //41974;              //GW port
            pResponse->Mode = 0x40;                     //Heartbeat, no timeout
            pResponse->Flags = 0x02;                    //TCP Connection
            pResponse->TimeOut = 0x0000;
            pResponse->Type = 0x0000;

            pResponse->HostName[0] = 0x00;

            pEntry = MasterListLockInst(0);
            pEntry = GetMasterListEntry(pEntry, pGatewayParams->myMID);
            if(pEntry)
            {
                //strncpy(pResponse->HostName, pEntry->SiteName, sizeof(pResponse->HostName));
                sprintf(pResponse->HostName,pEntry->SiteName);
            }
            MasterListUnlockInst(0);

            pResponse->MsgLength += strlen(pResponse->HostName) + 1;
            iDataLength = pResponse->MsgLength + 0x3;
            break;
        case eDISCONNECT :
            break;
        default:
            return -1;
    }
    // now let's send the data
    //NU_Obtain_Semaphore(&pGatewayParams->TransmitSemaphore,NU_SUSPEND);
    OsLock(pGatewayParams->TransmitSemaphore);
    //bytes_sent = NU_Send(pGatewayParams->gwSocket, pGatewayResponse, iDataLength, 0);

    // fixed endian issues
    pResponse->MsgLength = NATIVE2BE16(pResponse->MsgLength);
    pResponse->DestCID = NATIVE2BE16(pResponse->DestCID);
    pResponse->PortNumber = NATIVE2BE16(pResponse->PortNumber);
    pResponse->TimeOut = NATIVE2BE16(pResponse->TimeOut);
    pResponse->Type = NATIVE2BE16(pResponse->Type);

    bytes_sent = NetSendTcpData(pGatewayParams->gwSocket, pGatewayResponse, iDataLength);
    //NU_Release_Semaphore(&pGatewayParams->TransmitSemaphore);
    OsUnlock(pGatewayParams->TransmitSemaphore);

    if(bytes_sent < 0)
    {
        CloseGatewayConnection(pGatewayParams,ERROR_SEND_FAIL,bytes_sent);
    }
    else if(bytes_sent < iDataLength)
    {
        DmErrorPrintf("GWR: only sent %d of %d bytes\r\n", (int)bytes_sent, (int)iDataLength);
    }


    return bytes_sent;
}

/********************************************************
 * Function :  ProcessGatewayData
 ********************************************************/
void ProcessGatewayData(Gateway_Server_Params* pGatewayParams,char* buffer)
{
    MASTER_LIST_ENTRY* MasterPtr;
  UINT8 *pBuffer;
  UINT16 byteCnt;

    MasterPtr = MasterListLockInst(0);
    MasterPtr = GetMasterListEntry(MasterPtr, pGatewayParams->myMID);
    if(MasterPtr)
    {
        if (MasterPtr->LinkStatus != CIP_ONLINE )
        {
            CIPOnlineEvent(MasterPtr);

            MasterPtr->LinkStatus = CIP_ONLINE;
            MasterPtr->m_bUpdateRequestSent = 1;
            CheckForQueuedMessages(MasterPtr);

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
            DMMidpointControllerRegisterCipMaster(MasterPtr);
                                
#endif
        }
        // pass onward so that packet looks the same as the local cresnet packet
    // CIP Format:
    //     [0]       [1]   [2]      [3]       [4]       [5]       [6] ....
    //  <cip type> <len1> <len2> <handler1> <handler2> <data0>   <data> ...
    // where <data0> is the count of the number of bytes to follow and the start of a standard packet
    // get a memory block ...
    // get 3 extra bytes, one for header (length and source) and one a null terminator at end.
    byteCnt = buffer[5];
    if (MemMgr && (pBuffer = MemMgr->GetBlock(byteCnt+4)) != 0)
    {
      // copy message to buffer;  This buffer will be released after the command is
      // processed by the cresnet task.

      // Cresnet format          [0]    [1]    [2]    [3]    [4]
      //                      <source>  <id>  <cnt>  <type> [data]
      // copy the data + the count
      memcpy(pBuffer+CRESNET_BUF_LEN_OFFSET, (unsigned char*)&buffer[5], byteCnt+1);
      pBuffer[CRESNET_BUF_ID_OFFSET] = pGatewayParams->myMID;

      // set the source to be e CIP
      pBuffer[CRESNET_BUF_SRC_OFFSET] = NET_DEST_CIP;

      CresnetSlaveAcceptIncomingPacket(pBuffer);
    }
    // error buffer not available ...
    // for now just reset buffer and throw out message
  }
    MasterListUnlockInst(0);
}

/********************************************************
 * Function : SendDataToGateway
 ********************************************************/
int SendDataToGateway(MASTER_LIST_ENTRY* MasterNode,
        Gateway_Server_Params* pGatewayParams, UINT8 *pSrc, UINT32 byteCnt)
{
    static unsigned char OutputPacketBuff[DM_MAX_CIP_MSG_LEN + 5];

    pGATEWAY_DATA_MSG ResponsePtr;
    INT32 status;
    //long mouse_test;
    UINT16 MsgLength;

	if(pGatewayParams && pGatewayParams->TransmitSemaphore)
	{
		// need to protect static OutputPacketBuff
		OsLock(pGatewayParams->TransmitSemaphore);
	}
	else
	{
		DmErrorPrintf("GWS: Null pointer\r\n");
		return -1;
	}

    // build the "Cresnet Data" header
    ResponsePtr = (pGATEWAY_DATA_MSG)OutputPacketBuff;
    ResponsePtr->MsgType = eDATA;
    ResponsePtr->Handler = pGatewayParams->gwHandler;
    //ResponsePtr->MsgLength = MasterNode->ActivePacketLength + 2;  // add two bytes for Handler
    ResponsePtr->MsgLength = byteCnt + 2;  // add two bytes for Handler
    MsgLength = ResponsePtr->MsgLength;
    //MsgLength = byteCnt;
    //memcpy(&ResponsePtr->Data[0],MasterNode->ActiveOutputPacket, MasterNode->ActivePacketLength);  //2
    memcpy(&ResponsePtr->Data[0], pSrc, byteCnt);  //2

    AdjustAnalogPacket(MasterNode, (unsigned char *)&ResponsePtr->Data[0]); //2

    //mouse_test = test_for_mouse_packet(MasterNode, (unsigned char *)&ResponsePtr->Data[0]); //2
    //if (mouse_test != -1)
    //{
    //    MasterNode->mouse_flags[mouse_test] = 0;
    //    ResponsePtr->MsgLength = mouse_out_strings[mouse_test][1] + 1 + 2;  // add two bytes for CIP_ID
    //    memcpy(&ResponsePtr->Data[0],  &mouse_out_strings[mouse_test][1],
    //            mouse_out_strings[mouse_test][1] + 1);  //2
    //}

    //if(CresnetDebug & CIP_DEBUG)
    //{
    //    int i;
    //    char *cp = CresnetDebugMessage;
    //    unsigned char *pp = (unsigned char*)ResponsePtr;
    //
    //    cp += sprintf(cp,"GW Tx:");
    //    i = 0;
    //    do
    //    {
    //        cp += sprintf(cp," %02X",*(pp+i));
    //        i++;
    //    } while (i < ResponsePtr->MsgLength + 3);
    //   sprintf(cp,"\r\n");
    //    //WriteDebugData(DebugMessage);
    //    DmConsolePrintf(CresnetDebugMessage);
    //}

    ResponsePtr->Handler = NATIVE2BE16(ResponsePtr->Handler);
    ResponsePtr->MsgLength = NATIVE2BE16(ResponsePtr->MsgLength);

    status = NetSendTcpData(pGatewayParams->gwSocket, (UINT8 *)ResponsePtr, MsgLength + 3);
    OsUnlock(pGatewayParams->TransmitSemaphore);
    if (status < 0)
    {
        CloseGatewayConnection(pGatewayParams,ERROR_SEND_FAIL,status);
    }
    else if(status < ResponsePtr->MsgLength + 3)
    {
        DmErrorPrintf("GWS: only sent %lu of %lu bytes\r\n", (UINT32)status, (UINT32)ResponsePtr->MsgLength + 3);
    }

    //printf("GW : SendPacket %d bytes\r\n",status);

    //MasterNode->ActivePacketLength = 0;
    return 0;
}

/********************************************************
 * Function : TransmitGatewayData
 ********************************************************/
int TransmitPacket(MASTER_LIST_ENTRY* MasterNode, UINT8 *pSrc, UINT32 byteCnt)
{
    Gateway_Server_Params *gwThisServer = 0;
    int i, result = -1;
    MASTER_LIST_ENTRY* pMaster;
    MASTER_LIST_ENTRY* pMasterList;

    gwThisServer = gwServers;
    pMasterList = MasterListLockInst(0);
    for(i=0;i<GatewayConnectionCount;i++)
    {
        pMaster = GetMasterListEntry(pMasterList, gwThisServer->myMID);
        if(!pMaster)
        {
            continue;
        }
        if(pMaster == MasterNode)
        {
            result = SendDataToGateway(MasterNode, gwThisServer, pSrc, byteCnt);
            break;
        }
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
    MasterListUnlockInst(0);

    return result;
}


/********************************************************
 * Function : AddGatewayEntry
 ********************************************************/
UINT8 GatewayReconnectTaskExtra[OS_TASK_EXTRA_BYTES];

int AddGatewayEntry(MASTER_LIST_ENTRY* pEntry, int index)
{
    //NU_HOSTENT          HostEntry;
    Gateway_Server_Params *gwThisServer = 0, *gwPrevServer = gwServers;
    //void * pointer;
    //INT32 status;
    //UINT32 ticks;
    UINT32 param;
    UINT8 * extra = GatewayReconnectTaskExtra;

    //Insert the entry into the list
//    gwPrevServer = gwServers;
    if(GatewayConnectionCount>0)
    {
      // MNT - 03.09.09 - We don't support more than one entry for now
      // exit. Left the code in here in case we need to support
      // multiple IPT entries in the future

#if 0
        for(int i=0;i<GatewayConnectionCount;i++)
        {
            if(gwPrevServer->pNext)
                gwPrevServer = (Gateway_Server_Params*)gwPrevServer->pNext;
            else
                break;
        }
        //Create new entry
        gwThisServer = (Gateway_Server_Params *)OsAllocMemory(sizeof(Gateway_Server_Params));
#endif
    }
    else
    {
        gwThisServer = gwServers;
        gwPrevServer = 0;
    }

    if(!gwThisServer)
    {
        return -1;
    }
    //Zero out the struct
    memset(gwThisServer,0x00,sizeof(Gateway_Server_Params));

    //My Network Settings
    //*((UINT32*)gwThisServer->myIPaddress) = GetIPAddress();
    //gwThisServer->myCID            = CresnetGetID();

    //Remote Network address settings
    gwThisServer->gwCID         = ConvertTo16bitIpId(pEntry->CIP_ID);
    gwThisServer->gwDID         = pEntry->DeviceID;
    gwThisServer->gwSocket      = -1;
    gwThisServer->gotProgReady = 0;
    gwThisServer->cipStartupTimeout = 0;
    //if(pEntry->Flags & CIP_STORE_BY_NAME)

#ifndef EXTENDED_ETHERNET_SUPPORT
    if(!AutodiscoveryIsEnabled())
    {
        // only resolve domain name if we are NOT doing autodiscovery!
        //if (NU_Get_Host_By_Name (pEntry->SiteName, &HostEntry) == NU_SUCCESS)
        //    memcpy(&pEntry->IP_Address, HostEntry.h_addr, HostEntry.h_length);
        NetGetHostByName(pEntry->SiteName, &pEntry->IP_Address);
    }
#else
    // Resolve hostname using CDnsAutoDiscovery only if the hostname is given
    // Does not matter if host IP is found or not, the reconnect task will keep trying untill it succeeds
    if ( (g_pDnsAutoDiscovery != NULL) && (pEntry->SiteName[0] != 0) )
      g_pDnsAutoDiscovery->GetHostByName(pEntry->SiteName,
                                         sizeof(pEntry->SiteName),
                                         (UINT8*)&pEntry->IP_Address,
                                         sizeof(pEntry->IP_Address));
#endif
    memcpy((unsigned char*)&gwThisServer->gwIPaddress[0],(unsigned char*)&pEntry->IP_Address,4);

    gwThisServer->gwPort         = CIPGetPort();
    gwThisServer->gwStatus       = 0;

  // For now, force stream to 0 - currently implementation only supports 1 stream per device
  // more needs to be implemented.
  gwThisServer->bStream       = 0;

    gwThisServer->hbControl = HB_WAITING_FOR_CONNECT;
    gwThisServer->hbTimeout = 0;
    gwThisServer->hbPayload = 0;
    gwThisServer->hbTries   = 0;
    gwThisServer->myMID     = index;

    //List Pointers
    if(gwPrevServer)
        gwThisServer->pPrev      = (unsigned long*)gwPrevServer;
    gwThisServer->pNext          = 0;

    //Fill in pntr values in prev list entry
    if(gwPrevServer)
        gwPrevServer->pNext       = (unsigned long*)gwThisServer;
    //Increment List Counter
    GatewayConnectionCount++;

    //Create & start reconnect task for this connection
    //pointer = CrestAllocMemory(TASK_STACK_BYTES);
    //status = NU_Create_Task(&gwThisServer->ReconnectTask, "GTWRCON", Gateway_Reconnect_Task, (int)gwThisServer, NU_NULL,
    //        pointer, TASK_STACK_BYTES, TASK_PRIORITY, 0, NU_PREEMPT, NU_START);
    param = (UINT32)gwThisServer;

#ifdef CRESNET_GATEWAY_DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
    {
        DmConsolePrintf("Creating reconnect task...\r");
    }
#endif
    OsCreateNamedPriorityTask(Gateway_Reconnect_Task, 5000, 0, &param, extra, "GwR", basicwebGATEWAY_PRIORITY, 128);

    //if(status != NU_SUCCESS)
    //{
    //    DmErrorPrintf("GATEWAY: Could not create reconnect task");
    //    return -1;
    //}

    //status = NU_Create_Semaphore(&gwThisServer->TransmitSemaphore, "GWTXSMP", 1, NU_FIFO);
    //if(status != NU_SUCCESS)
    //{
    //    DmSystemError (FATAL_APPINIT);
    //}
    OsCreateLock(&gwThisServer->TransmitSemaphore);

#ifdef CIP_STARTUP_TIMER
    //ticks = HwMsToTicks(GW_CIP_STARTUP_MSEC);
    //if(NU_Create_Timer(&gwThisServer->timer, "TGW", GatewayCipStartupTimerExpiration,
    //        (UNSIGNED)gwThisServer, ticks, 0,
    //        NU_DISABLE_TIMER) != NU_SUCCESS)
    //{
    //    DmErrorPrintf("GATEWAY: Could not create timer!");
    //}
    gwThisServer->timer = OsStartTimer(GW_CIP_STARTUP_MSEC, GatewayCipStartupTimerExpiration, 0);
    //}
    //else
    //    return -1;
#endif
    return 0;
}

/**
 * \author      Pete McCormick (copied from Jeff Paige)
 *
 * \date        3/6/2006
 *
 * \return      MASTER_LIST_ENTRY *
 *
 * \retval      Pointer to corresponding master list entry
 *
 * \brief       Get the master list entry with the specified handle
 *
 * \param       pMasterList - pointer to master list
 * \param       handle
 *
 * \warning     Must lock master list before calling this function.
 *
 */
MASTER_LIST_ENTRY * GetMasterListEntry(MASTER_LIST_ENTRY* pMasterList, unsigned long handle)
{
    MASTER_LIST_ENTRY * TravPtr;

    // search through the list
    TravPtr = pMasterList;
    while(TravPtr)
    {
        if(handle == TravPtr->handle)
        {
            break;
        }
        TravPtr = TravPtr->Next;
    }
    return TravPtr;
}

/********************************************************
 * Function : SendGatewayConnectUpdateRequest
 ********************************************************/
int SendGatewayConnectUpdateRequest(MASTER_LIST_ENTRY* NewConnection, Gateway_Server_Params* pGatewayParams)
{
    char cPacket[DM_MAX_CIP_MSG_LEN];
    pGATEWAY_DATA_MSG pResponse;
    int status;
    UINT16 MsgLength;

    if(!(EthLinkStatusOK()))  // if no link, return;
    {
        DmErrorPrintf("GWAY: can't send update request: ethernet link down\r\n");
        return -1;
    }

    //DmConsolePrintf("SendGatewayConnectUpdateRequest\n");

    memset(cPacket,0,DM_MAX_CIP_MSG_LEN);          // clear message.

    pResponse = (pGATEWAY_DATA_MSG)cPacket;
    pResponse->MsgType = eDATA;
    pResponse->MsgLength = 5;
    MsgLength = pResponse->MsgLength;
    pResponse->Handler = pGatewayParams->gwHandler;

    //*(unsigned short *)pResponse->Data = NewConnection->CIP_ID;  // fill in CIP ID
    pResponse->Data[0] = 0x02;        // Update request length
    pResponse->Data[1] = CNET_COMMAND;     // Update request Cresnet packet type
    if(CresnetSupportSelectiveAllClear())
        pResponse->Data[2] = HELPER_SELECTIVE_ALLCLEAR_REQUEST;
    else
        pResponse->Data[2] = UPDATE_REQUEST;

    if (GetNodeSemaphore(NewConnection) )
    {
        NewConnection->InTransNumber = 0;
        NewConnection->OutTransNumber = 0;
        ReleaseNodeSemaphore(NewConnection);
    }

    //NU_Obtain_Semaphore(&pGatewayParams->TransmitSemaphore,NU_SUSPEND);
    OsLock(pGatewayParams->TransmitSemaphore);

    pResponse->MsgLength = NATIVE2BE16(pResponse->MsgLength);
    pResponse->Handler = NATIVE2BE16(pResponse->Handler);

    //status = NU_Send(pGatewayParams->gwSocket, (char *)pResponse, pResponse->MsgLength + 3, 0);
    status = NetSendTcpData(pGatewayParams->gwSocket, (UINT8*)pResponse, MsgLength + 3);

    //NU_Release_Semaphore(&pGatewayParams->TransmitSemaphore);
    OsUnlock(pGatewayParams->TransmitSemaphore);

    if (status < 0)
    {
        DmErrorPrintf("Error sending Update Request Packet\r\n");
        return -1;
    }
    else if(status < pResponse->MsgLength + 3)
    {
        DmErrorPrintf("GWUR: only sent %u of %u bytes\r\n", status, pResponse->MsgLength + 3);
    }
    else
    {
        if (GetNodeSemaphore(NewConnection) )
        {
            NewConnection->m_bUpdateRequestSent = 1;
            ReleaseNodeSemaphore(NewConnection);
        }
        if (!CresnetSupportSelectiveAllClear())
        {
#ifdef NEEDED
			g_bAllClearFilter = 1;
            g_iUpdateRequestCntr++;
#endif //NEED
            //status = NU_Control_Timer (&AllClearTimer,  NU_DISABLE_TIMER);
            OsCancelTimer(AllClearTimer);

            //if (status == NU_SUCCESS)
            //{
            //    status = NU_Reset_Timer (&AllClearTimer, ResetAllClearFilter,
            //            ALL_CLEAR_TIMER_FILTER_TIMEOUT, 0, NU_ENABLE_TIMER);
            //    if (status != NU_SUCCESS)
            //    {
            //        DmErrorPrintf("Error could not reset AllClearTimer: %d", status);
            //    }
            //}
            //else
            //{
            //    DmErrorPrintf("Error could not disable AllClearTimer: %d", status);
            //}
        }
    }
    return 0;
}

INT32 GatewayTasksInit(void)
{
#ifdef CRESNET_GATEWAY_DEBUG
#ifdef DEBUG
    if(CresnetIsDebug(CRESNET_DEBUG_CIP_INIT))
    {
        DmConsolePrintf("Creating gateway task...\r");
    }
#endif // DEBUG
#endif
    Gateway_Task(0);

    return 0;
}

/**
* \author      Larry Salant
* \date        7/19/2011
* \brief       sets the lower bits of the i6 bit IP ID for this device
* \param       Ip Id
* \return      void
*/
void GatewayTasksAdvancedInit(UINT16 ipId16)
{
	IpId16bit = ipId16;
	GatewayTasksInit();
}

void GatewayUpdateEntry(MASTER_LIST_ENTRY * pMasterListEntry, BOOL hostnameChanged)
{
    Gateway_Server_Params *gwThisServer;
    UINT32 i;

    gwThisServer = gwServers;
    for(i=0; i<GatewayConnectionCount; i++)
    {
        if(gwThisServer->myMID == pMasterListEntry->handle)
        {
            gwThisServer->gwCID = ConvertTo16bitIpId(pMasterListEntry->CIP_ID);
            gwThisServer->gwDID = pMasterListEntry->DeviceID;
            if(hostnameChanged)
            {
			    memcpy((unsigned char*)&gwThisServer->gwIPaddress[0],(unsigned char*)&pMasterListEntry->IP_Address,4);
                // need to kill current TCP connection
                CloseGatewayConnection(gwThisServer, ERROR_MASTER_LIST_CHANGE, 0);
            }
            return;
        }
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
}
#ifdef DEBUG
void Gateway_Print_Info(void)
{
    Gateway_Server_Params *gwThisServer;
    UINT32 i;
    char ipAddrTxt[32];
    char * hbCtrlTxt[] = {"waiting for connect", "waiting for resend",
            "waiting for response", "waiting for connection", "terminated"};
    char hbCtrlTxtTemp[32];
    MASTER_LIST_ENTRY* pEntry;
    UINT32 ipAddr;

    gwThisServer = gwServers;
    for(i=0; i<GatewayConnectionCount; i++)
    {
        ipAddr = *((unsigned long *)(gwThisServer->gwIPaddress));
        ConvertIPAddressToString(ipAddr, ipAddrTxt);
        if(gwThisServer->hbControl < sizeof(hbCtrlTxt)/sizeof(hbCtrlTxt[0]))
        {
            strncpy(hbCtrlTxtTemp, hbCtrlTxt[gwThisServer->hbControl], sizeof(hbCtrlTxtTemp));
        }
        else
        {
            snprintf(hbCtrlTxtTemp, sizeof(hbCtrlTxtTemp), "%lu", gwThisServer->hbControl);
        }

        // for autodiscovery, ip addr will be zero in ip table
        if(AutodiscoveryIsEnabled() && (ipAddr == 0))
        {
            ipAddrTxt[0] = 0;
            pEntry = MasterListLockInst(0);
            pEntry = GetMasterListEntry(pEntry, gwThisServer->myMID);
            if(pEntry)
            {
                strncpy(ipAddrTxt, pEntry->SiteName, sizeof(ipAddrTxt));
            }
            MasterListUnlockInst(0);
        }

        DmConsolePrintf("\r");
        DmConsolePrintf("Socket..............: %lu\r", gwThisServer->gwSocket);
        DmConsolePrintf("Server..............: %s\r", ipAddrTxt);
        DmConsolePrintf("Status..............: %s\r", gwThisServer->gwStatus ? "connected" : "disconnected");
        DmConsolePrintf("Heartbeat control...: %s\r", hbCtrlTxtTemp);
        //DmConsolePrintf("Heartbeat timeout...: %ld\r", gwThisServer->hbTimeout);
        //DmConsolePrintf("Heartbeat retries...: %u\r", gwThisServer->hbTries);
#ifdef GATEWAY_STATS
        DmConsolePrintf("Heartbeats sent.....: %lu\r", gwThisServer->hbSentCnt);
        DmConsolePrintf("Heartbeats received.: %lu\r", gwThisServer->hbRcvCnt);
        DmConsolePrintf("Heartbeat send errs.: %lu\r", gwThisServer->hbSendErrCnt);
        DmConsolePrintf("Heartbeat rcv errs..: %lu\r", gwThisServer->hbRcvErrCnt);
        DmConsolePrintf("Server ip errs......: %lu\r", gwThisServer->gwServIpErrCnt);
        DmConsolePrintf("Connection attempts.: %lu\r", gwThisServer->connAttempts);
        DmConsolePrintf("Connection errors...: %lu\r", gwThisServer->connErrCnt);
        DmConsolePrintf("Connection count....: %lu\r", gwThisServer->connCnt);
#endif
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }

}
#endif // DEBUG

void Gateway_Reset_Counters(void)
{
    Gateway_Server_Params *gwThisServer;
    UINT32 i;

    gwThisServer = gwServers;
    for(i=0; i<GatewayConnectionCount; i++)
    {
#ifdef GATEWAY_STATS
        gwThisServer->hbSentCnt = 0;
        gwThisServer->hbRcvCnt = 0;
        gwThisServer->hbSendErrCnt = 0;
        gwThisServer->hbRcvErrCnt = 0;
        gwThisServer->connAttempts = 0;
        gwThisServer->connErrCnt = 0;
        gwThisServer->connCnt = 0;
        gwThisServer->gwServIpErrCnt = 0;
#endif
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }
}

/**
 * \author      Unknown
 *
 * \date        04/09/2007
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handles error condition when there is a timeout writing to the
 *              transmit pipe.
 *
 * \param       NodePtr
 *
 * \note        This should never really happen.
 */
void ResetCIP_Node(MASTER_LIST_ENTRY * NodePtr)
{
    //INT32 status;
    Gateway_Server_Params *gwThisServer;
    UINT32 i;
    //UINT32 item;

    // find the corresponding gateway server params struct
    gwThisServer = gwServers;
    for(i=0; i<GatewayConnectionCount; i++)
    {
        if(gwThisServer->myMID == NodePtr->handle)
        {
            break;
        }
        gwThisServer = (Gateway_Server_Params*)gwThisServer->pNext;
    }

    if(i >= GatewayConnectionCount)
    {
        // could not find a match
        return;
    }

    if (!GetNodeSemaphore(NodePtr) )
    {
        return;
    }

    //NodePtr->TransmitState = 0;
    //NodePtr->ActivePacketLength = 0;  // clear active packet
    //if(NodePtr->NumOutputMsgsQueued > 0)
    //{
    //    //status = NU_Reset_Pipe(NodePtr->OutputPipe);
    //    status = 0;
    //    while(status == 0)
    //    {
    //        status = OsDequeue(NodePtr->OutputPipe, &item );
    //    }
    //    //if ( status != NU_SUCCESS)
    //    //{
    //    //    DmErrorPrintf("Node Reset: Error reseting pipe");
    //    //}
    //    NodePtr->NumOutputMsgsQueued = 0;
    //}
    ReleaseNodeSemaphore(NodePtr);

    // now close the gateway connection
    CloseGatewayConnection(gwThisServer, ERROR_TX_PIPE_TIMEOUT, 0);
}
/**
* \author      Larry Salant
* \date        8/14/2008
* \brief       check if the gateway for this stream is on line and available
* \param       void
* \return      UINT8
* \retval      1 if Gateway is available
*/
UINT8 IsGatewayAvailable(UINT8 destination)
{
  MASTER_LIST_ENTRY* CIP_target = (MASTER_LIST_ENTRY*) FindMasterByDeviceID(destination);

  if (CIP_target && CIP_target->LinkStatus == CIP_ONLINE)
    return true;
  else
    return false;

}
int AnyGatewayConnections()
{
	return GatewayConnectionCount;
}
/**
* \author      Larry Salant
* \date        7/18/2011
* \brief       converts the 8 bit IP ID to a 16 bit IP ID if needed
* \param       8bit Ip Id
* \return      UINT16
* \retval      16 bit IP ID
*/
UINT16 ConvertTo16bitIpId(UINT16 IpId8Bit)
{
	if (IpId16bit == NO_16BIT_IPID)
		return IpId8Bit;
	else
		return (IpId8Bit << 8 | IpId16bit);
}